import { world, system } from "@minecraft/server";
import { cachedPlayers } from './pvp.js';

// 1) On player join: if they already have gwim:VIP, add VIPPurchased tag
world.afterEvents.playerJoin.subscribe((event) => {
  const player = event.player;
  if (player.hasTag("gwim:VIP") && !player.hasTag("VIPPurchased")) {
    player.addTag("VIPPurchased");
  }
});

// Predefined zones to exclude from effect logic
const exclusionZones = [
  [835, -60, -1214, 1235, 200, -814],
  [-910, 4, 919, -1097, 200, 1088],
  [-2974, 3, 2987, -2936, 20, 3013],
];

const RANK_TAGS = ["gwim:VIP", "gwim:VIPPlus", "gwim:Elite", "gwim:ElitePlus", "gwim:God"];

// 2) Every tick: effects + VIP tag enforcement
system.runInterval(() => {
  const players = cachedPlayers;
  if (players.length === 0) return;

  for (const player of players) {
    const { x, y, z } = player.location;

    const isExcluded = player.dimension.id === "minecraft:the_end" ||
      exclusionZones.some(([x1, y1, z1, x2, y2, z2]) =>
        x >= Math.min(x1, x2) && x <= Math.max(x1, x2) &&
        y >= Math.min(y1, y2) && y <= Math.max(y1, y2) &&
        z >= Math.min(z1, z2) && z <= Math.max(z1, z2)
      );

    if (isExcluded) continue;

    // Apply effects only if missing or wrong amplifier
    const effects = [
      { id: "minecraft:weakness", duration: 50, amplifier: 255 },
      { id: "minecraft:fire_resistance", duration: 50, amplifier: 255 },
      { id: "minecraft:resistance", duration: 50, amplifier: 255 },
      { id: "minecraft:instant_health", duration: 50, amplifier: 255 },
    ];

    for (const { id, duration, amplifier } of effects) {
      const current = player.getEffect(id);
      if (!current || current.amplifier !== amplifier) {
        try {
          player.addEffect(id, duration, { amplifier, showParticles: false });
        } catch (e) {
          console.warn(`Failed to apply ${id} to ${player.name}: ${e}`);
        }
      }
    }

    try {
      const tagSet = new Set(player.getTags());
      const hasVIPPurchased = tagSet.has("VIPPurchased");
      const hasVIP = tagSet.has("gwim:VIP");

      const hasHigherRank =
        tagSet.has("gwim:VIPPlus") ||
        tagSet.has("gwim:Elite") ||
        tagSet.has("gwim:ElitePlus") ||
        tagSet.has("gwim:God");

      if (hasVIPPurchased) {
        if (!hasVIP && !hasHigherRank) {
          player.addTag("gwim:VIP");
        }
      } else {
        if (hasVIP) player.removeTag("gwim:VIP");
        for (const tag of RANK_TAGS.slice(1)) {
          if (tagSet.has(tag)) player.removeTag(tag);
        }
      }
    } catch (err) {
      console.warn(`Error processing VIP for ${player.name}: ${err}`);
    }
  }
}, 1); // Every tick

// 3) Auto-upgrade logic (every second)
system.runInterval(() => {
  const moneyObj = world.scoreboard.getObjective("Money");
  if (!moneyObj) return;

  const players = cachedPlayers;
  if (players.length === 0) return;

  for (const player of players) {
    try {
      const identity = player.scoreboardIdentity;
      let money = 0;
      try {
        money = moneyObj.getScore(identity);
      } catch {
        money = 0;
      }

      const tagSet = new Set(player.getTags());
      const hasVIPPurchased = tagSet.has("VIPPurchased");

      if (money <= 80000000) {
        for (const tag of RANK_TAGS.slice(1)) {
          if (tagSet.has(tag)) player.removeTag(tag);
        }
        continue;
      }

      if (!hasVIPPurchased) continue;

      let targetTag = "";
      let targetRank = "";

      if (money >= 1_000_000_000) {
        targetTag = "gwim:God";
        targetRank = "§d§l|| GOD ||";
      } else if (money >= 400_000_000) {
        targetTag = "gwim:ElitePlus";
        targetRank = "§b§lElite+";
      } else if (money >= 200_000_000) {
        targetTag = "gwim:Elite";
        targetRank = "§b§lElite";
      } else {
        targetTag = "gwim:VIPPlus";
        targetRank = "§e§lVIP+";
      }

      if (!tagSet.has(targetTag)) {
        for (const tag of RANK_TAGS) {
          if (tag !== targetTag && tagSet.has(tag)) {
            player.removeTag(tag);
          }
        }
        player.addTag(targetTag);
        player.playSound("random.levelup");
        player.sendMessage(`§aYour rank is now ${targetRank}`);
      }
    } catch (err) {
      console.warn(`Error in auto-rank for ${player.name}: ${err}`);
    }
  }
}, 20); // Every second

// 4) VIP-only zone check (every 5 seconds)
const ZONE_CENTER = { x: 4028, y: 5, z: 4000 };
const ZONE_RADIUS = 60;
const ZONE_RADIUS_SQ = ZONE_RADIUS * ZONE_RADIUS;
const allowedTags = new Set([
  "gwim:VIP", "gwim:VIPPlus", "gwim:Elite", "gwim:ElitePlus", "gwim:God", "gwim:Admin"
]);

function isInZone(pos, center) {
  const dx = pos.x - center.x;
  const dz = pos.z - center.z;
  return dx * dx + dz * dz <= ZONE_RADIUS_SQ;
}

system.runInterval(() => {
  const players = cachedPlayers;
  if (players.length === 0) return;

  for (const player of players) {
    const pos = player.location;
    const tagSet = new Set(player.getTags());

    const hasPermission = [...allowedTags].some(tag => tagSet.has(tag));
    const inZone = isInZone(pos, ZONE_CENTER);

    if (inZone && !hasPermission && !tagSet.has("Lobby")) {
      player.addTag("Lobby");
      player.sendMessage("§cYou don't have any VIP or higher ranks");
    }
  }
}, 100); // Every 5 seconds